import subprocess
import requests
import base64
import os

from flask import Flask, render_template, request, jsonify, send_from_directory

app = Flask(__name__)

# Ajusta la URL de tu webhook n8n o el que uses
WEBHOOK_URL = "TU WEBHOOK"


@app.route('/')
def index():
    """
    Página principal: carga la plantilla HTML (index.html)
    """
    return render_template('index.html')


@app.route('/chat', methods=['POST'])
def chat():
    user_message = request.form.get('message', '')

    # 1. Enviar el texto al webhook
    payload = {"text": user_message}
    response = requests.post(WEBHOOK_URL, json=payload, timeout=120)

    if response.status_code != 200:
        return jsonify({
            "error": True,
            "message": "Error al enviar petición al webhook"
        }), 500

    # 2. Obtener la respuesta del webhook
    try:
        webhook_data = response.json()
    except Exception:
        return jsonify({
            "error": True,
            "message": f"Error al parsear la respuesta del webhook: {response.text}"
        }), 500

    respuesta_chat = webhook_data.get("respuesta", "")
    # Aquí asumimos que 'codigo_diagrama' es base64
    codigo_diagrama_b64 = webhook_data.get("codigo_diagrama", "")

    # 3. Decodificar el contenido base64 (si existe)
    if codigo_diagrama_b64:
        try:
            # Decodificamos base64 a texto (asumiendo UTF-8)
            codigo_diagrama = base64.b64decode(codigo_diagrama_b64).decode('utf-8')
        except Exception as e:
            return jsonify({
                "error": True,
                "message": f"Error al decodificar el diagrama en base64: {str(e)}"
            }), 500

        # Guardar el YAML decodificado en template.yaml
        with open("template.yaml", "w", encoding="utf-8") as f:
            f.write(codigo_diagrama)

        # Ejecutar awsdac
        cmd = ["awsdac", "-v", "template.yaml", "-o", "static/diagrama.png"]
        try:
            subprocess.run(cmd, capture_output=True, text=True, check=True)
        except subprocess.CalledProcessError as e:
            return jsonify({
                "error": True,
                "message": f"Error al generar diagrama: {e.stderr}"
            }), 500

    # 4. Leer el contenido de template.yaml para enviarlo al front
    template_content = ""
    try:
        with open("template.yaml", "r", encoding="utf-8") as f:
            template_content = f.read()
    except FileNotFoundError:
        pass

    return jsonify({
        "error": False,
        "respuesta": respuesta_chat,
        "template_content": template_content
    })

@app.route('/update-template', methods=['POST'])
def update_template():
    """
    Endpoint para actualizar manualmente el contenido de template.yaml
    desde el editor en la web, y volver a generar la imagen.
    """
    new_content = request.form.get('template_content', '')

    # Guardar el nuevo contenido en template.yaml
    with open("template.yaml", "w", encoding="utf-8") as f:
        f.write(new_content)

    # Ejecutar awsdac para regenerar la imagen
    cmd = ["awsdac", "-v", "template.yaml", "-o", "static/diagrama.png"]
    try:
        subprocess.run(cmd, capture_output=True, text=True, check=True)
    except subprocess.CalledProcessError as e:
        return jsonify({
            "error": True,
            "message": f"Error al generar diagrama: {e.stderr}"
        }), 500

    return jsonify({
        "error": False,
        "message": "Diagrama actualizado"
    })


@app.route('/get-template-content', methods=['GET'])
def get_template_content():
    """
    (Opcional) Endpoint para obtener el contenido de template.yaml
    al cargar la página (si se quiere mostrar antes de interactuar con el chat).
    """
    template_content = ""
    try:
        with open("template.yaml", "r", encoding="utf-8") as f:
            template_content = f.read()
    except FileNotFoundError:
        pass

    return jsonify({
        "error": False,
        "template_content": template_content
    })


@app.route('/static/<path:filename>')
def custom_static(filename):
    """
    Sirve archivos estáticos (por ejemplo, diagrama.png).
    """
    return send_from_directory('static', filename)


if __name__ == '__main__':
    # Para desarrollo local: flask run (o python app.py)
    # Cambia host/puerto según necesidades
    app.run(debug=True, host='0.0.0.0', port=5000)
